// ************************************************************************************************
//
// Zeitschaltuhr unter Verwendung des
//
// PIC 16F627A
//
// Ausgabe der Zeit ber drei LED-Displays,
// zustzliche (statische) Ausgabe ber zwei permanent leuchtende Doppelpunkte
//
// Trigger: Fallende Flanke an RB0 (INT)
//			(entspricht steigender Flanke am Optokoppler-Eingang,
//			da der CNY17 in dieser Schaltung invertiert)
//
// Nach Triggerung: Zhler zhlt von 0:00 aufwrts und Anzeige beginnt ab 1:30 zu blinken
// (nderbar durch Variable "SekundenMax" (diese entspricht der Anzahl der Sekunden, also hier 90)
//
// Triggersensitivitt einstellbar durch Variablen "TriggerMinLength" und "TriggerExitMinGap"
//
// Copyright www.jb-electronics.de
//
// ************************************************************************************************

// ************************************************************************************************
// PIC-Header einbinden
#include <pic.h>

// ************************************************************************************************
// Methodenprototypen
uchar get7SegmentValue (uchar Wert);
void updateDisplay ();
void calculate ();

// ************************************************************************************************
// Globale Variablen
static int TimingCounter, BlinkCounter, MuxCounter, Sekunden, SekundenMax, TicksSinceTriggered, TriggerMinLength, TicksSinceNoSignal, TriggerExitMinGap;
static uchar SEG1, SEG2, SEG3, Sekunden_E, Sekunden_Z, Minuten_E, Blink, Active, Triggered, toDisplay;

// ************************************************************************************************
// Hauptmethode
void main (void) {

    // ********************************************************************************************
	// Konfigurationsbits setzen
	__CONFIG(0x3F2A);

	// ********************************************************************************************
	// Portspezifikationen
	#define	TRIGGER					RB0
	#define SEGMENTE				PORTB
	#define ANODE_SEKUNDEN_EINER	RA0
	#define ANODE_SEKUNDEN_ZEHNER	RA1
	#define ANODE_MINUTEN_EINER		RA2

	// Tristate von PORTB setzen
	TRISB = 0b00000001;

	// RA0...RA2 als Ausgang setzen und Analogkomparator abschalten
	TRISA = 0b000;
	CMCON = 0b111;

	// ********************************************************************************************
	// TIMER setzen
	T0CS = 0;
	PSA = 1;
	PS0 = 1;
	PS1 = 1;
	PS2 = 1;

	// ********************************************************************************************
	// Interrupt bei Timer-berlauf auslsen
	T0IE = 1;

	// ********************************************************************************************
	// Externen Interrupt bei fallender Flanke an RB0 einstellen
	INTE = 1;
	INTEDG = 0;

	// Interrupts aktivieren
	GIE = 1;

	// ********************************************************************************************
	// Maximalzeit einstellen, hier: 90s i.e. 1:30 Minuten
	SekundenMax = 90;

	// ********************************************************************************************
	// Ticks, die das Triggersignal anliegen muss, um als solches gewertet zu werden
	// (1s entspricht ca. 4096 Ticks)
	TriggerMinLength = 450;

	// ********************************************************************************************
	// Ticks, die das Triggersignal weggenommen werden muss, um als solches gewertet zu werden
	// (1s entspricht ca. 4096 Ticks)
	TriggerExitMinGap = 450;

	// ********************************************************************************************
	// Variablen resetten
	Active = 0;
	Triggered = 0;
	Sekunden = 0;
	Sekunden_E = 0;	SEG1 = 0b110000000;
	Sekunden_Z = 0;	SEG2 = 0b110000000;
	Minuten_E = 0;	SEG3 = 0b110000000;

	// ********************************************************************************************
	// Beginn der Hauptschleife
	while (true) {

	}

}

// ************************************************************************************************
// Interrupt-Routine fr genauen 1Hz-Takt
static void interrupt isr (void) {

	// ********************************************************************************************
	// ein bisschen Zeit verstrichen?
	if (T0IF) {

		// Blinkfunktion
		BlinkCounter++;
		if (BlinkCounter >= 1023) {
			Blink = 1 - Blink;
			BlinkCounter = 0;
		}
		if ((Blink == 0) && (Sekunden >= SekundenMax) && (TRIGGER == 0)) {
			toDisplay = 0;
			MuxCounter = 0;
		}

		// Zeit anzeigen
		updateDisplay ();

		// Multiplex-Verteiler: Wann und wie lange ist welche Anzeige "dran"?
		MuxCounter++;
		if (MuxCounter > 15) {
			MuxCounter = 0;
			toDisplay++;
			if (toDisplay > 3) {
				toDisplay = 1;
			}
		}

		// eine Sekunde um?
		TimingCounter++;
		if (TimingCounter == 4095) {
			if (Active == 1) {
				Sekunden++;
				Sekunden_E++;
				if (Sekunden_E >= 10) {
					Sekunden_E = 0;
					Sekunden_Z++;
					if (Sekunden_Z >= 6) {
						Sekunden_Z = 0;
						Minuten_E++;
					}
				}
				TimingCounter = 0;
			}
			calculate();
		}

		// Triggersignal lange genug da?
		if (TRIGGER == 0) {

			// Das Signal muss mindestens #TriggerMinLength Takte anliegen
			if (TicksSinceTriggered < TriggerMinLength) {
				TicksSinceTriggered++;

			// Signal lange genug da
			} else {

				// Wenn noch nicht aktiv, dann resetten
				if (Active == 0) {

					TicksSinceNoSignal = 0;
					TimingCounter = 0;
					BlinkCounter = 0;
					Sekunden = 0;
					Sekunden_E = 0;
					Sekunden_Z = 0;
					Minuten_E = 0;
					Active = 1;
					calculate ();
				}

			}
		
		// Trigger-Signal weg?
		} else {

			if (TicksSinceNoSignal <= TriggerExitMinGap) {
				TicksSinceNoSignal++;
			} else {

				TicksSinceTriggered = 0;
				Active = 0;

			}

		}

		// Interrupt resetten
		T0IF = 0;

	}

}

// ************************************************************************************************
// wandelt einen Wert in ein Zeichen im 7-Segment-Code um
uchar get7SegmentValue (uchar Wert) {

	// Variablen
	uchar result;

	// welches Ausgabemuster bei welchem Zeichen?
	//
	//   Bit    Segment
	//
	//    0       a
	//    1       b
	//    2       c
	//    3       d
	//    4       f
	//    5       g
	//    6       h
	//    7       DP
	//
	if (Wert == 0) {
		result = 0b11000000;
	} else if (Wert == 1) {
		result = 0b11111001;
	} else if (Wert == 2) {
		result = 0b10100100;
	} else if (Wert == 3) {
		result = 0b10110000;
	} else if (Wert == 4) {
		result = 0b10011001;
	} else if (Wert == 5) {
		result = 0b10010010;
	} else if (Wert == 6) {
		result = 0b10000010;
	} else if (Wert == 7) {
		result = 0b11111000;
	} else if (Wert == 8) {
		result = 0b10000000;
	} else if (Wert == 9) {
		result = 0b10010000;
	} else {
		result = 0b10001110;
	}

	// Wert zurckgeben
	// WICHTIG: um ein Bit nach links verschoben, da Segment a bei RB1, und NICHT bei RB0.
	return (result << 1);

}

// ************************************************************************************************
// Diese Methode zeigt den Countdownstand an
void updateDisplay () {

	// alles aus
	SEGMENTE = 0b11111110;
	ANODE_SEKUNDEN_EINER = 1;
	ANODE_SEKUNDEN_ZEHNER = 1;
	ANODE_MINUTEN_EINER = 1;
	
	if (toDisplay == 1) {
		SEGMENTE = SEG1;
		ANODE_SEKUNDEN_EINER = 0;
	} else if (toDisplay == 2) {
		SEGMENTE = SEG2;
		ANODE_SEKUNDEN_ZEHNER = 0;
	} else if (toDisplay == 3) {
		SEGMENTE = SEG3;
		ANODE_MINUTEN_EINER = 0;
	}

}

// ************************************************************************************************
// Diese Methode errechnet die Segmentmuster fr das Display
// (aufgrund von Performance-Einbuen wird diese Methode nur einmal pro Sekunde ausgefhrt)
void calculate () {

	SEG1 = get7SegmentValue(Sekunden_E);
	SEG2 = get7SegmentValue(Sekunden_Z);
	SEG3 = get7SegmentValue(Minuten_E);

}
