// ************************************************************************************************
//
// LED-Digitalzhler unter Verwendung des
//
// PIC 16F627A
//
// Ausgabe des Zhlerstands ber drei LED-Displays
//
// Incrementieren / Resetten des Zhlers ber zwei Taster
//
// Automatisches Puffern des Zhlsignals auf 1s
//
// Copyright www.jb-electronics.de
//
// ************************************************************************************************

// ************************************************************************************************
// PIC-Header einbinden
#include <pic.h>

// ************************************************************************************************
// Methodenprototypen
void warte ();
uchar get7SegmentValue (uchar Wert, uchar DecimalPoint);
void schreibeWert (uchar Wert, uchar Wohin);
void updateDisplay ();

// ************************************************************************************************
// Globale Variablen
static int Counter, TimeBuffer;

// ************************************************************************************************
// Hauptmethode
void main (void) {

    // ********************************************************************************************
	// Konfigurationsbits setzen
	__CONFIG(0x3F2A);

	// ********************************************************************************************
	// Portspezifikationen
	#define	DATA	RB0
	#define CLK		RB1
	#define SEG1	RB2
	#define SEG2	RB3
	#define SEG3	RB4
	#define INC		RB5
	#define RESET	RB6

	// Tristate setzen
	TRISB0 = 0;
	TRISB1 = 0;
	TRISB2 = 0;
	TRISB3 = 0;
	TRISB4 = 0;
	TRISB5 = 1;
	TRISB6 = 1;

	// ********************************************************************************************
	// Zhler resetten
	Counter = 0;

	// ********************************************************************************************
	// Zeitpuffer resetten (wobei 1s vergangen = ~3906 Ticks bei 4MHz Takt)
	TimeBuffer = 0;

	// ********************************************************************************************
	// TIMER setzen
	T0CS = 0;
	PSA = 1;
	PS0 = 1;
	PS1 = 1;
	PS2 = 1;

	// Interrupt bei Timer-berlauf auslsen
	T0IE = 1;

	// Interrupts aktivieren
	GIE = 1;

	// ********************************************************************************************
	// Beginn der Hauptschleife
	while (true) {

		// Zhlerstand anzeigen
		updateDisplay ();

		// Incrementieren? Dabei TimeBuffer beachten!
		if ((INC) && (TimeBuffer == 0)) {
			Counter++;
			while (INC) {
				updateDisplay ();
				TimeBuffer = 3906;
			}
		}

		// Resetten?
		if (RESET) {
			Counter = 0;
			while (RESET) {
				updateDisplay ();
			}
		}

	}

}

// ************************************************************************************************
// Interrupt-Routine fr genauen 1Hz-Takt
static void interrupt isr (void) {

	// ********************************************************************************************
	// TimeBuffer decrementieren
	if (T0IF) {
		if (TimeBuffer > 0) {
			TimeBuffer--;
		}
		T0IF = 0;
    }

}

// ************************************************************************************************
// Diese Methode wartet kurz, um Signale gltig werden zu lassen
void warte () {

	// einen Takt lang nichts tun
	NOP();

}

// ************************************************************************************************
// schreibt einen Wert in ein Schieberegister
void schreibeWert (uchar Wert, uchar Wohin) {

	// Variablen
	int Index;

	// initialisieren
	SEG1 = 0;
	SEG2 = 0;
	SEG3 = 0;

	// erst die Zehner
	for (Index = 7; Index >= 0; Index--) {
		warte();
		if (Wert & (1 << Index)) {
			DATA = 1;
		} else {
		 	DATA = 0;
		}
		warte();
		CLK = 1;
		warte();
		CLK = 0;
	}

	// Wert bernehmen
	if (Wohin == 1) {
		SEG1 = 1;
	} else if (Wohin == 2) {
		SEG2 = 1;
	} else if (Wohin == 3) {
		SEG3 = 1;
	}
	warte();

	// zurcksetzen
	SEG1 = 0;
	SEG2 = 0;
	SEG3 = 0;

}

// ************************************************************************************************
// wandelt einen Wert in ein Zeichen im 7-Segment-Code um
uchar get7SegmentValue (uchar Wert, uchar DecimalPoint) {

	// Variablen
	uchar result;

	// welches Ausgabemuster bei welchem Zeichen?
	//
	//   Bit    Segment
	//
	//    0       a
	//    1       b
	//    2       c
	//    3       d
	//    4       f
	//    5       g
	//    6       h
	//    7       DP
	//
	if (Wert == 0) {
		result = 0b00111111;
	} else if (Wert == 1) {
		result = 0b00000110;
	} else if (Wert == 2) {
		result = 0b01011011;
	} else if (Wert == 3) {
		result = 0b01001111;
	} else if (Wert == 4) {
		result = 0b01100110;
	} else if (Wert == 5) {
		result = 0b01101101;
	} else if (Wert == 6) {
		result = 0b01111101;
	} else if (Wert == 7) {
		result = 0b00000111;
	} else if (Wert == 8) {
		result = 0b01111111;
	} else if (Wert == 9) {
		result = 0b01101111;
	} else {
		result = 0b01110001;
	}

	// Dezimalpunkt aktiv?
	if (DecimalPoint) {
		result| = 0b10000000;
	}
	
	// Wert zurckgeben
	return (result);

}

// ************************************************************************************************
// Diese Methode zeigt den Countdownstand an
void updateDisplay () {

	// Zhlerstand anzeigen

	// Hunderter
	schreibeWert(get7SegmentValue(Counter / 100, 0), 3);

	// Zehner
	schreibeWert(get7SegmentValue(((Counter - Counter % 10 - ((Counter / 100) * 100)) / 10), 0), 2);

	// Einer
	schreibeWert(get7SegmentValue(Counter % 10, 0), 1);

}
