// ************************************************************************************************
//
// LED Alarm Clock using the
//
// PIC16F716
//
// Copyright www.jb-electronics.de
//
// http://www.jb-electronics.de/html/elektronik/digital/d_wecker.htm
//
// ************************************************************************************************

// ************************************************************************************************
// PIC-Header einbinden
#include <pic.h>

// ************************************************************************************************
// method prototypes
void updateDisplay ();
void waitSomeTime ();
void waitSomeLongTime ();

// ************************************************************************************************
// global variables
static int t, timebase, STD_BUFFER, MIN_BUFFER, SLP_BUFFER, SLP_ON_TIME, MP3_ON_TIME, ALARM_ON_TIME = 0, ALARM_ON_TIME_MAX;
static unsigned char i, j, pwm_index, adc_count, adc_tmp, Stunden, Minuten, Sekunden, bitmuster;
static unsigned char WeckStunden, WeckMinuten, WeckStundenNext, WeckMinutenNext, WeckSekundenNext;
static unsigned char displayValue[4];
static unsigned char get7[10] = {0b01110111, 0b00010001, 0b10110110, 0b10110011, 0b11010001, 0b11100011, 0b11100111, 0b00110001, 0b11110111, 0b11110011};
static bit MIN, SLP, ARMED;
static int MIN_BUFFER, STD_BUFFER, SLP_BUFFER;
static unsigned char SleepMinutes = 5;
static unsigned char Modus = 0;
static unsigned char MP3_phase = 0;
static int pwm_table[16]= {0, 1, 3, 5, 10, 20, 50, 80, 100, 150, 200, 300, 400, 600, 800, 1023};

// ************************************************************************************************
// main method
void main (void) {

    // ********************************************************************************************
	// configuration bit
	__CONFIG(0x3f29);

	// ********************************************************************************************
	// port specifications
    #define DATA	RB0
    #define CLK		RB1
    #define STROBE	RB2
    #define PWM		RB3
    #define COLON	RB4
	#define MP3		RB5
	#define LIGHT	RB6
	#define FAN		RB7
	TRISB = 0b00000000;

    #define ADC		RA0
    #define	SLP_SW	RA2
    #define MIN_SW	RA3
	#define	STD		RA4
	TRISA = 0b11111;


	// *****
	// ADC settings:

	// ADC sampling time per bit is 8 * T_osc
	ADCS0 = 1;
	ADCS1 = 0;

	// RA0:3 ports as analog inputs
	PCFG0 = 0;
	PCFG1 = 0;
	PCFG2 = 0;

	// turn it ON
	ADON = 1;

	// wait some time
	for (t = 0; t <= 100; t++) {
		NOP ();
	}


	// *****
	// configure Timer0 as a timer with no prescaler, interrupt on overflow
	T0CS = 0;
	PSA = 1;
	T0IE = 1;

	// enable global interrupts
	GIE = 1;


	// *****
	// PWM settings

	// enable PWM mode
	CCP1M0 = 0;
	CCP1M1 = 0;
	CCP1M2 = 1;
	CCP1M3 = 1;

	// only RB3 as an output, all other pins as digital IOs
	P1M0 = 0;
	P1M1 = 0;

	
	// PWM duty cycle

	// (two LSB of 10 Bit value)
	DC1B0 = 0;
	DC1B1 = 0;

	// (eight MSB if 10 Bit value)
	CCPR1L = 0;

	// upper limit of Timer2
	PR2 = 0xff;

	// set prescaler of Timer2 to 1:1
	T2CKPS0 = 0;
	T2CKPS1 = 0;

	// turn it on
	TMR2ON = 1;


	// *****
	// (p)reset variables and ports
	MP3 = 0;
	LIGHT = 0;
	COLON = 0;
	FAN = 0;

	// time
	Stunden = 12;
	Minuten = 0;

	// alarm time (default: 6:30)
	WeckStunden= 6;
	WeckMinuten = 30;

	// SLEEP interval: 5 minutes
	SleepMinutes = 5;

	// maximum ALARM ON TIME: 4 minutes
	ALARM_ON_TIME_MAX = 240;

	// *****
	// main loop
	while (1) {


		// *****
		// read SLP (RA2) and MIN (RA3) buttons
		CHS0 = 0;
		CHS1 = 1;
		CHS2 = 0;
		GO = 1;
		while (GO);
		if (ADRES > 127) {
			SLP = 1;
		} else {
			SLP = 0;
		}
		CHS0 = 1;
		CHS1 = 1;
		CHS2 = 0;
		GO = 1;
		while (GO);
		if (ADRES > 127) {
			MIN = 1;
		} else {
			MIN = 0;
		}
	

		// *****
		// fan control
		if (CCPR1L > 32) {
			FAN = 1;
		} else {
			FAN = 0;
		}


		// *****
		// normal clock mode, waiting for the alarm
		if (Modus == 0) {


			// read and adjust brightness using the potentiometer on RA0
			CHS0 = 0;
			CHS1 = 0;
			CHS2 = 0;
			GO = 1;
			while (GO);
			adc_tmp += ADRES >> 4;
			adc_count++;
			if (adc_count >= 4) {
				adc_count = 0;
				pwm_index = adc_tmp >> 2;
				adc_tmp = 0;
			}
			CCPR1L = (pwm_table[pwm_index] & 0b1111111100) >> 2;
			DC1B0 = pwm_table[pwm_index] & 0b01;
			DC1B1 = (pwm_table[pwm_index] & 0b10) >> 1;

			// react on user inputs

			// hours
			if ((STD != 0) && (STD_BUFFER == 0)) {
	
				// sleep pressed?
				if (SLP) {
					WeckStunden++;
					if (WeckStunden >= 24) {
						WeckStunden = 0;
					}
				
				// normal hour setting
				} else {
					Stunden++;
					if (Stunden >= 24) {
						Stunden = 0;
					}
					Sekunden = 0;
					timebase = 0;
	
				}
	
				// refill buffer
				STD_BUFFER = 1000;

				// delete continous on-time for SLEEP switch
				SLP_ON_TIME = 0;
	
			}
	
			// minutes
			if ((MIN != 0) && (MIN_BUFFER == 0)) {
	
				// sleep pressed?
				if (SLP) {
					WeckMinuten++;
					if (WeckMinuten >= 60) {
						WeckMinuten = 0;
					}
				
				// normal hour setting
				} else {
					Minuten++;
					if (Minuten >= 60) {
						Minuten = 0;
					}
					Sekunden = 0;
					timebase = 0;
	
				}
	
				// refill buffer
				MIN_BUFFER = 1000;

				// delete continous on-time for SLEEP switch
				SLP_ON_TIME = 0;
	
			}

			// SLEEP (you need to hold it about four seconds without any other action to toggle the mode
			if (SLP != 0) {
	
				// toggle flag?
				if (SLP_ON_TIME > 12500) {
				
					ARMED ^= 1;
					SLP_ON_TIME = 0;

				}
	
				// refill buffer
				SLP_BUFFER = 1000;
	
			}


			// first alarm reached?
			if ((ARMED) && (Minuten == WeckMinuten) && (Stunden == WeckStunden)) {

				// prepare the sleep variables
				WeckMinutenNext = WeckMinuten;
				WeckStundenNext = WeckStunden;

				// start the alarm mode
				Modus = 1;
		
				// the alarm has not been on at all
				ALARM_ON_TIME = 0;

				// to turn on the MP3 player, we need a two second signal on port MP3
				// do this over a variable, we need to do it here.
				MP3_ON_TIME = 10000;

			}


		// *****
		// alarm mode
		} else if (Modus == 1) {


			// dim up the light
			if (CCPR1L < 0xff) {
				t = CCPR1L;
				t += 5;
				if (t > 0xff) {
					t = 0xff;
				}
				CCPR1L = t;
			}

			// is it fully dimmed up?
			if (CCPR1L == 0xff) {
			
				// turn on the police light (the MP3 player is already being turned on in ISR)
				LIGHT = 1;

			}


			// react on user inputs

			//      any key will work as SLEEP
			// OR   security shutdown when alarm is active for more than ALARM_ON_TIME_MAX
			if ((((SLP != 0) && (SLP_BUFFER == 0)) || ((MIN != 0) && (MIN_BUFFER == 0)) || ((STD != 0) && (STD_BUFFER == 0))) || (ALARM_ON_TIME >= ALARM_ON_TIME_MAX))  {

				// change the mode to 2, i.e. sleep mode
				Modus = 2;

				// turn off the police light
				LIGHT = 0;
				
				// turn off MP3 player
				MP3_ON_TIME = 0;
				MP3_phase = 1;

				// go back to sleep. But before, calculate the next waketime using the SleepMinutes variable
				WeckSekundenNext = Sekunden;
				WeckMinutenNext = Minuten + SleepMinutes;
				if (WeckMinutenNext >= 60) {
					WeckMinutenNext -= 60;
					WeckStundenNext += 1;
					if (WeckStundenNext >= 24) {
						WeckStundenNext -= 24;
					}
				}

				// reset alarm on time
				ALARM_ON_TIME = 0;

				// refill the buffers
				SLP_BUFFER = 8000;
				MIN_BUFFER = 8000;
				STD_BUFFER = 8000;

			}


		// *****
		// sleep mode
		} else if (Modus == 2) {


			// read and adjust brightness using the potentiometer on RA0
			CHS0 = 0;
			CHS1 = 0;
			CHS2 = 0;
			GO = 1;
			while (GO);
			adc_tmp += ADRES >> 4;
			adc_count++;
			if (adc_count >= 4) {
				adc_count = 0;
				pwm_index = adc_tmp >> 2;
				adc_tmp = 0;
			}
			CCPR1L = (pwm_table[pwm_index] & 0b1111111100) >> 2;
			DC1B0 = pwm_table[pwm_index] & 0b01;
			DC1B1 = (pwm_table[pwm_index] & 0b10) >> 1;
			

			// alarm feature
			if ((ARMED) && (Stunden == WeckStundenNext) && (Minuten == WeckMinutenNext) && (Sekunden == WeckSekundenNext)) {

				// to turn on the MP3 player, we need a two second signal on port MP3
				// do this over a variable, we need to do it here.
				MP3_ON_TIME = 10000;

				// go to alarm mode
				Modus = 1;

			}

			
			// react on user inputs
			if (SLP != 0) {

				// in order to turn OFF the alarm, hold SLEEP for about four seconds
				if (SLP_ON_TIME > 12500) {
				
					ARMED = 0;
					SLP_ON_TIME = 0;
					Modus = 0;

				}

				// refill buffer
				SLP_BUFFER = 1000;

			}


		}


		// *****
		// update the display
		updateDisplay ();


	}


}

// ************************************************************************************************
// interrupt service routine
static void interrupt isr (void) {

	// Timer0 overflow?
	if (T0IF) {

		// extract 1Hz information
		timebase++;
		if (timebase >= 4096) {

			// how long has the alarm been switched on?
			if (Modus == 1) {
				ALARM_ON_TIME += 1;
			}

			// extract 1Hz
			timebase = 0;
			Sekunden++;
			if (Sekunden >= 60) {
				Sekunden = 0;
				Minuten++;
				if (Minuten >= 60) {
					Minuten = 0;
					Stunden++;
					if (Stunden >= 24) {
						Stunden = 0;
					}
				}
			}

		}

		// buffer switches
		if (STD_BUFFER > 0) {
			STD_BUFFER--;
		}
		if (MIN_BUFFER > 0) {
			MIN_BUFFER--;
		}
		if (SLP_BUFFER > 0) {
			SLP_BUFFER--;
		}

		// calculate on time
		if (SLP != 0) {
			SLP_ON_TIME++;
		}

		// control the MP3 player

		if (MP3_phase == 0) {

			if (MP3_ON_TIME > 0) {
				MP3 = 1;
				MP3_ON_TIME--;
			} else {
				MP3_ON_TIME = 0;
				MP3 = 0;
			}
		
		// short PAUSE pulse
		} else if (MP3_phase == 1) {
		
			MP3_ON_TIME++;
			if (MP3_ON_TIME <= 500) {
				MP3 = 1;
			} else if (MP3_ON_TIME <= 1000) {
				MP3 = 0;
			} else {
				MP3_ON_TIME = 0;
				MP3_phase = 2;
			}

		} else if (MP3_phase == 2) {

			MP3_ON_TIME++;
			if (MP3_ON_TIME <= 4000) {
				MP3 = 0;
			} else if (MP3_ON_TIME <= 8000) {
				MP3 = 1;
			} else {
				MP3 = 0;
				MP3_ON_TIME = 0;
				MP3_phase = 3;
			}

		} else if (MP3_phase == 3) {

			MP3_ON_TIME++;
			if (MP3_ON_TIME <= 4000) {
				MP3 = 0;
			} else if (MP3_ON_TIME <= 8000) {
				MP3 = 1;
			} else {
				MP3 = 0;
				MP3_ON_TIME = 0;
				MP3_phase = 0;
			}
				

		}

		// reset interrupt flag
		T0IF = 0;

	}

}

// ************************************************************************************************
// this method updates the display
void updateDisplay () {

	STROBE = 0;
	
	// decimal point
	unsigned char dp_armed = 0;

	if ((SLP) && (Modus == 0)) {
		displayValue[0] = WeckMinuten % 10;
		displayValue[1] = WeckMinuten / 10;
		displayValue[2] = WeckStunden % 10;
		displayValue[3] = WeckStunden / 10;
	} else if ((SLP) && (Modus == 2)) {
		displayValue[0] = WeckMinutenNext % 10;
		displayValue[1] = WeckMinutenNext / 10;
		displayValue[2] = WeckStundenNext % 10;
		displayValue[3] = WeckStundenNext / 10;
	} else {
		displayValue[0] = Minuten % 10;
		displayValue[1] = Minuten / 10;
		displayValue[2] = Stunden % 10;
		displayValue[3] = Stunden / 10;
	}

	// main loop
	for (i = 0; i <= 3; i++) {

		if ((ARMED) && (i == 0)) {
			dp_armed = 0b1000;
		} else {
			dp_armed = 0;
		}

		bitmuster = get7[displayValue[i]] | dp_armed;
		for (j = 0; j <= 7; j++) {
			DATA = (bitmuster & (1 << j)) >> j;
			CLK = 1;
			waitSomeTime ();
			CLK = 0;
			waitSomeTime ();
		}

	}

	STROBE = 1;
	waitSomeTime ();
	STROBE = 0;

}

// ************************************************************************************************
void waitSomeTime () {

	NOP ();	NOP ();	NOP ();	NOP ();	NOP ();
	NOP ();	NOP ();	NOP ();	NOP ();	NOP ();
	NOP ();	NOP ();	NOP ();	NOP ();	NOP ();
	NOP ();	NOP ();	NOP ();	NOP ();	NOP ();

}

// ************************************************************************************************
void waitSomeLongTime () {

	for (t = 0; t < 5000; t++) {
		waitSomeTime ();
	}

}