// ************************************************************************************************
//
// Copyright www.jb-electronics.de
//
// Sport-Display - Anzeigeplatine
//
// (x) Empfang der Daten ber die USART-Schnittstelle
// (x) Anzeige der Daten ber Ausgabe an Schieberegister
// (x) Automatisches Abschalten der LEDs nach einer Empfangspause von einer Sekunde (siehe Zeile 27)
//
// http://www.jb-electronics.de/html/elektronik/digital/d_sport_display.htm
//
// ************************************************************************************************


// ************************************************************************************************
// PIC-Header einbinden
#include <pic.h>


// ************************************************************************************************
// Globale Variablen
static unsigned char value = 0, address = 0;
static int i = 0;
static const unsigned char ADDRESS_FLAG = 0b10000000;
static long idle_count = 0;

// 1s Abschaltdauer; 4096 Ticks = 1s bei Quarz von 4.194304MHz
static const long idle_max = 4096;

// Hier werden die Segment-Daten gespeichert
static unsigned char time_data[4] = {0, 0, 0, 0};
static unsigned char score_data[4] = {0, 0, 0, 0};
static unsigned char thirds_data = 0;
static unsigned char colon_upper = 0;
static unsigned char colon_lower = 0;


// ************************************************************************************************
// Hauptmethode
void main (void) {


    // *****
	// Konfigurationsbits setzen
	__CONFIG(FOSC_XT & WDTE_OFF & PWRTE_ON & MCLRE_ON & BOREN_ON & LVP_OFF & CPD_OFF & CP_OFF);


	// *****
	// TRISTATE-Register setzen, sowie Pindefinitionen
	
	// RB1 = RX
	TRISB1 = 1;

	// RB2 = TX
	TRISB2 = 0;

	// Datenausgnge
	#define DATA			RB0
	TRISB0 = 0;
	#define CLK				RB3
	TRISB3 = 0;
	#define STROBE_TIME		RB4
	TRISB4 = 0;
	#define STROBE_SCORE1	RB7
	TRISB7 = 0;
	#define STROBE_SCORE2	RB5
	TRISB5 = 0;
	#define STROBE_THIRDS	RB6
	TRISB6 = 0;


	// *****
	// USART auf RS232 konfigurieren
	
	// schnelle Baud-Rate
	BRGH = 1;

	// Baud-Rate einstellen: 128000
	// F_OSC = 4.194304MHz, Baud-Rate = F_OSC / 16 / (SPBRG + 1)
	SPBRG = 1;

	// USART-Modus auf "asynchron" stellen
	SYNC = 0;

	// die serielle Schnittstelle einschalten
	SPEN = 1;

	// Interrupt bei empfangenem Zeichen auslsen
	RCIE = 1;

	// Empfangsmodus einschalten
	CREN = 1;


	// *****
	// TIMER0 einschalten
	T0CS = 0;
	PSA = 1;
	PS0 = 1;
	PS1 = 1;
	PS2 = 1;
	T0IE = 1;


	// *****
	// Interrupts aktivieren

	// global
	GIE = 1;

	// periphere Interrupts aktivieren (der USART-Interrupt ist ein peripherer Interrupt!)
	PEIE = 1;


	// *****
	// Beginn der Hauptschleife
	while (1) {


		// Zeitdaten senden
		for (i = 7; i >= 0; i--) {
			DATA = (time_data[3] & (1 << i)) >> i;
			CLK = 1;
			CLK = 0;
		}
		for (i = 7; i >= 0; i--) {
			DATA = ((time_data[2] | colon_upper) & (1 << i)) >> i;
			CLK = 1;
			CLK = 0;
		}
		for (i = 7; i >= 0; i--) {
			DATA = ((time_data[1] | colon_lower) & (1 << i)) >> i;
			CLK = 1;
			CLK = 0;
		}
		for (i = 7; i >= 0; i--) {
			DATA = (time_data[0] & (1 << i)) >> i;
			CLK = 1;
			CLK = 0;
		}
		STROBE_TIME = 1;
		STROBE_TIME = 0;


		// Spielstandsdaten senden
		for (i = 7; i >= 0; i--) {
			DATA = (score_data[3] & (1 << i)) >> i;
			CLK = 1;
			CLK = 0;
		}
		for (i = 7; i >= 0; i--) {
			DATA = (score_data[2] & (1 << i)) >> i;
			CLK = 1;
			CLK = 0;
		}
		STROBE_SCORE1 = 1;
		STROBE_SCORE1 = 0;
		

		// Spielstandsdaten senden
		for (i = 7; i >= 0; i--) {
			DATA = (score_data[1] & (1 << i)) >> i;
			CLK = 1;
			CLK = 0;
		}
		for (i = 7; i >= 0; i--) {
			DATA = (score_data[0] & (1 << i)) >> i;
			CLK = 1;
			CLK = 0;
		}
		STROBE_SCORE2 = 1;
		STROBE_SCORE2 = 0;


		// Dritteldaten senden
		for (i = 7; i >= 0; i--) {
			DATA = (thirds_data & (1 << i)) >> i;
			CLK = 1;
			CLK = 0;
		}
		STROBE_THIRDS = 1;
		STROBE_THIRDS = 0;


	}


}

// ************************************************************************************************
// Interrupt-Service-Routine (ISR)
static void interrupt isr (void) {

	// Zeichen empfangen?
	if (RCIF) {

		// Wert empfangen; setze die Variable f&uuml;r den automatischen TImeout daher auf null
		idle_count = 0;

		// Wert auslesen
		value = RCREG;

		// eventuelle Fehler clearen
		// (hier sollte eigentlich eine richtige Fehlerberprfung eingebaut sein)
		CREN = 0;
		CREN = 1;
		OERR = 0;
		FERR = 0;

		// war das Zeichen eine Adresse?
		if (value & ADDRESS_FLAG) {
			address = (value & 0b01111111);			
		
		// ansonsten waren es Daten, die nun an die vorher bermittelte Adresse gesetzt werden
		} else {

			value &= 0b01111111;
			if (address == 1) {
				time_data[0] = value;
			} else if (address == 2) {
				time_data[1] = value;
			} else if (address == 3) {
				time_data[2] = value;
			} else if (address == 4) {
				time_data[3] = value;
			} else if (address == 5) {
				score_data[0] = value;
			} else if (address == 6) {
				score_data[1] = value;
			} else if (address == 7) {
				score_data[2] = value;
			} else if (address == 8) {
				score_data[3] = value;
			} else if (address == 9) {
				thirds_data = value;
			} else if (address == 10) {
				if (value != 0) {
					colon_upper = 0b10000000;
				} else {
					colon_upper = 0;
				}
			} else if (address == 11) {
				if (value != 0) {
					colon_lower = 0b10000000;
				} else {
					colon_lower = 0;
				}
			}

		}

		RCIF = 0;

	}

	// TIMER0-berlauf?
	if (T0IF) {

		// es passiert gerade nichts
		if (idle_count < idle_max) {
			idle_count++;

		// wenn idle_max Ticks nichts empfangen wurde, gehe davon aus, dass die Fernbedienung abgeklemmt wurde, und schalte das Display aus.
		} else {

			time_data[0] = 0;
			time_data[1] = 0;
			time_data[2] = 0;
			time_data[3] = 0;
			score_data[0] = 0;
			score_data[1] = 0;
			score_data[2] = 0;
			score_data[3] = 0;
			thirds_data = 0;
			colon_upper = 0;
			colon_lower = 0;

		}

		// Interrupt-Flag resetten
		// (muss beim RCIF von oben nicht (!) gemacht werden, dieses wird automatisch beim Auslesen des RCREG gecleart)
		T0IF = 0;


	}

}