// ************************************************************************************************
//
// Copyright www.jb-electronics.de
//
// Sport-Display - Fernbedienung
//
// (x) Auswertung aller Eingabeinstrumente (acht Taster, ein Schalter, zwei Drehschalter)
// (x) Ansteuern von fnf Siebensegment-Displays im Multiplex-Betrieb
// (x) Senden der Displaydaten an die Gesamtdisplays unter Nutzung des USART-Moduls
//
// http://www.jb-electronics.de/html/elektronik/digital/d_sport_display.htm
//
// ************************************************************************************************

// ************************************************************************************************
// PIC-Header einbinden
#include <pic.h>

unsigned char getTime ();

// ************************************************************************************************
// Globale Variablen
static int i = 0, timebase = 0, mux_timebase = 0, alarm_timebase = 0, blink_timebase = 0;
static const unsigned char ADDRESS_FLAG = 0b10000000;
static const int alarm_time = 4;
static unsigned char MinutenE = 0, MinutenZ = 0, SekundenE = 0, SekundenZ = 0, ScoreHeimE = 0, ScoreHeimZ = 0, ScoreGastE = 0, ScoreGastZ = 0, Drittel = 0, upper_colon = 1, lower_colon = 1, DrittelAlt = 0;
static unsigned char display_min = 0, display_sec = 0, display_heim = 0, display_gast = 0, display_drittel = 0, mux_status = 0, display_score = 0;
static unsigned char sirene = 0, running = 0, tmp = 0, firstrun = 1;
static int SW_START_STOP_BUFFER, SW_RESET_BUFFER, SW_HEIM_PLUS_BUFFER, SW_HEIM_MINUS_BUFFER, SW_GAST_PLUS_BUFFER, SW_GAST_MINUS_BUFFER, SW_SCORE_RESET_BUFFER, SW_SIRENE_BUFFER;

// Daten, die zum Gesamtdisplay gesendet werden, und Laufindex what_to_send
static unsigned char send_data[23];
static unsigned char what_to_send = 0;

// Sieben-Segment-Muster
static const unsigned char get7[10] = {0b00111111,
									   0b00000110,
									   0b01011011,
									   0b01001111,
									   0b01100110,
									   0b01101101,
									   0b01111101,
									   0b00000111,
									   0b01111111,
									   0b01101111};

// ************************************************************************************************
// Hauptmethode
void main (void) {


    // *****
	// Konfigurationsbits setzen (XT, WDT disabled, PWRT enabled, MCLR enabled, Code Protetction Off, Brown-Out-Detect enabled, Brown-Out-Reset-Voltage bei 2.5V, INTOSC-Frequenz = 500kHz)
	__CONFIG(0x2be1);


	// *****
	// TRISTATE-Register setzen, sowie Pindefinitionen

	// RC7 = RX
	TRISC7 = 1;
	ANSC7 = 0;

	// RC6 = TX
	TRISC6 = 0;
	ANSC6 = 0;

	// Taster
	#define SW_START_STOP		RB7
	#define	SW_RESET			RB6
	#define SW_HEIM_PLUS		RB5
	#define SW_HEIM_MINUS		RB4
	#define SW_GAST_PLUS		RB1
	#define SW_GAST_MINUS		RB0
	#define SW_SCORE_RESET		RB2
	#define SW_SIRENE			RB3
	TRISB = 0b11111111;
	ANSB0 = 0; ANSB1 = 0; ANSB2 = 0; ANSB3 = 0; ANSB4 = 0; ANSB5 = 0; ANSB6 = 0; ANSB7 = 0;
	
	// interne Pullups einschalten
	nRBPU = 0;

	// Der Drehschalter fr die Spielzeit
	#define TIME_1		RC0
	#define TIME_2		RC1
	#define TIME_3		RD7
	#define TIME_4		RD6
	#define TIME_5		RD5
	#define TIME_6		RD4
	#define TIME_7		RC2
	#define TIME_8		RC3
	#define TIME_9		RC5
	#define TIME_10		RC4
	#define TIME_11		RD3
	#define TIME_12		RD2
	TRISC0 = 1; TRISC1 = 1; TRISD7 = 1; TRISD6 = 1; TRISD5 = 1; TRISD4 = 1; TRISC2 = 1; TRISC3 = 1; TRISC5 = 1; TRISC4 = 1; TRISD3 = 1; TRISD2 = 1;
	ANSC0 = 0; ANSC1 = 0; ANSD7 = 0; ANSD6 = 0; ANSD5 = 0; ANSD4 = 0; ANSC2 = 0; ANSC5 = 0; ANSD3 = 0; ANSD2 = 0;

	// Der Drehschalter fr das Drittel
	#define DRITTEL_A	RD0
	#define DRITTEL_B	RD1
	TRISD0 = 1; TRISD1 = 1;
	ANSD0 = 0; ANSD1 = 0;

	// Der Schalter fr die Sirene
	#define SIRENE_AUTO	RA1
	TRISA1 = 1;
	ANSA1 = 0;

	// Der Schlsselschalter
	#define MASTER_EIN	RA2
	TRISA2 = 1;
	ANSA2 = 0;

	// Der Anschluss der Schieberegister
	#define DATA		RA3
	#define CLOCK		RA5
	#define STROBE		RA4
	TRISA3 = 0; TRISA5 = 0; TRISA4 = 0;
	ANSA3 = 0; ANSA5 = 0; ANSA4 = 0;
	
	// Die Multiplex-Steuerleitungen
	#define MUX_LEFT	RE0
	#define MUX_RIGHT	RE1
	TRISE0 = 0; TRISE1 = 0;
	ANSE0 = 0; ANSE1 = 0;

	// Das Relais
	#define RELAIS		RA0
	TRISA0 = 0;
	ANSA0 = 0;

	// Der Doppelpunkt
	#define COLON	RE2
	TRISE2 = 0;
	ANSE2 = 0;


	// *****
	// USART auf RS422 konfigurieren
	// (exakt gleiche Parameter wie bei RS232; heit hier nur RS422 wegen hardwaremig anderer bertragung)
	
	// schnelle Baud-Rate
	BRGH = 1;

	// Baud-Rate einstellen: 128000
	// F_OSC = 4.194304MHz, Baud-Rate = F_OSC / 16 / (SPBRG + 1)
	SPBRG = 1;

	// USART-Modus auf "asynchron" stellen
	SYNC = 0;

	// die serielle Schnittstelle einschalten
	SPEN = 1;

	// Interrupt bei empfangenem Zeichen auslsen
	//RCIE = 1;

	// Empfangsmodus einschalten
	CREN = 1;

	// Sendemodus einschalten
	TXEN = 1;


	// *****
	// Interrupts aktivieren

	// global
	GIE = 1;

	// periphere Interrupts aktivieren (der USART-Interrupt ist ein peripherer Interrupt!)
	PEIE = 1;


	// *****
	// Timer0-berlauf bei 4094 Ticks einstellen
	T0CS = 0;
	PSA = 1;
	PS0 = 1;
	PS1 = 1;
	PS2 = 1;
	T0IE = 1;


	// *****
	// Variablen resetten
	MinutenE = getTime() % 10;
	MinutenZ = getTime() / 10;
	SekundenE = 0; SekundenZ = 0;
	ScoreHeimE = 0; ScoreHeimZ = 0;
	ScoreGastE = 0; ScoreGastZ = 0;

	// Sendedaten vorbereiten
	// (die Adressen werden nur hier einmal gesetzt, diese ndern sich ja nicht im Betrieb)
	send_data[0] = ADDRESS_FLAG | 0b0001;
	send_data[1] = get7[SekundenE];
	send_data[2] = ADDRESS_FLAG | 0b0010;
	send_data[3] = get7[SekundenZ];
	send_data[4] = ADDRESS_FLAG | 0b0011;
	send_data[5] = get7[MinutenE];
	send_data[6] = ADDRESS_FLAG | 0b0100;
	send_data[7] = get7[MinutenZ];	
	send_data[8] = ADDRESS_FLAG | 0b1010;
	send_data[9] = upper_colon;
	send_data[10] = ADDRESS_FLAG | 0b1011;
	send_data[11] = lower_colon;
	send_data[12] = ADDRESS_FLAG | 0b0101;
	send_data[13] = get7[ScoreHeimE];
	send_data[14] = ADDRESS_FLAG | 0b0110;
	send_data[15] = get7[ScoreHeimZ];
	send_data[16] = ADDRESS_FLAG | 0b0111;
	send_data[17] = get7[ScoreGastE];
	send_data[18] = ADDRESS_FLAG | 0b1000;
	send_data[19] = get7[ScoreGastZ];
	send_data[20] = ADDRESS_FLAG | 0b1001;
	send_data[21] = get7[Drittel];
	send_data[22] = 0;

	// Doppelpunkt einschalten
	COLON = 1;

	// Spielzeit pausieren
	running = 0;

	// Spielstand soll angezeigt werden
	display_score = 1;


	// *****
	// Beginn der Hauptschleife
	while (1) {

		
		// Ist die Fernbedienung ber den Schlsselschalter berhaupt eingeschaltet?
		if (MASTER_EIN) {


			// Drittel-Schalter auswerten; Zwischenpositionen abfangen; aber nicht bei erstem Durchlauf!
			DrittelAlt = Drittel;
			Drittel = (DRITTEL_A ^ 1) | ((DRITTEL_B ^ 1) << 1);
			if (Drittel == 3) {
				if (DrittelAlt == 0) {
					if (firstrun == 0) {
						Drittel = 0;
					}
				}
				if (DrittelAlt == 1) {
					Drittel = 1;
				}
			}
			firstrun = 0;
			if (Drittel != 0) {
				send_data[21] = get7[Drittel];
			} else {
				send_data[21] = 0;
			}


			// Taster-Abfrage mit eingebauter Entprellung bzw. Mindest-Drckzeit
			if (SW_START_STOP == 0) {
				if (SW_START_STOP_BUFFER == 0) {
					running ^= 1;
					timebase = 0;
					blink_timebase = 0;
					SW_START_STOP_BUFFER = 70;
				}
			}
			if (SW_RESET == 0) {
				if (SW_RESET_BUFFER >= 40) {
					running = 0;
					tmp = getTime();
					MinutenE = tmp % 10;
					MinutenZ = tmp / 10;
					SekundenE = 0;
					SekundenZ = 0;
					timebase = 0;
					alarm_timebase = 0;
					sirene = 0;
				} else {
					SW_RESET_BUFFER++;
				}
			} else {
				SW_RESET_BUFFER = 0;
			}
			if (SW_HEIM_PLUS == 0) {
				if (SW_HEIM_PLUS_BUFFER == 0) {
					if (ScoreHeimE < 9) {
						ScoreHeimE++;
					} else if (ScoreHeimZ < 9) {
						ScoreHeimE = 0;
						ScoreHeimZ++;
					}
					send_data[13] = get7[ScoreHeimE];
					send_data[15] = get7[ScoreHeimZ];
					SW_HEIM_PLUS_BUFFER = 40;
				}
			}
			if (SW_HEIM_MINUS == 0) {
				if (SW_HEIM_MINUS_BUFFER == 0) {
					if (ScoreHeimE > 0) {
						ScoreHeimE--;
					} else if (ScoreHeimZ > 0) {
						ScoreHeimE = 9;
						ScoreHeimZ--;
					}
					send_data[13] = get7[ScoreHeimE];
					send_data[15] = get7[ScoreHeimZ];
					SW_HEIM_MINUS_BUFFER = 40;
				}
			}
			if (SW_GAST_PLUS == 0) {
				if (SW_GAST_PLUS_BUFFER == 0) {
					if (ScoreGastE < 9) {
						ScoreGastE++;
					} else if (ScoreGastZ < 9) {
						ScoreGastE = 0;
						ScoreGastZ++;
					}
					send_data[17] = get7[ScoreGastE];
					send_data[19] = get7[ScoreGastZ];
					SW_GAST_PLUS_BUFFER = 40;
				}
			}
			if (SW_GAST_MINUS == 0) {
				if (SW_GAST_MINUS_BUFFER == 0) {
					if (ScoreGastE > 0) {
						ScoreGastE--;
					} else if (ScoreGastZ > 0) {
						ScoreGastE = 9;
						ScoreGastZ--;
					}
					send_data[17] = get7[ScoreGastE];
					send_data[19] = get7[ScoreGastZ];
					SW_GAST_MINUS_BUFFER = 40;
				}
			}
			if (SW_SCORE_RESET == 0) {
				if (SW_SCORE_RESET_BUFFER >= 250) {
					display_score ^= 1;
					if (display_score == 0) {
						send_data[13] = 0;
						send_data[15] = 0;
						send_data[17] = 0;
						send_data[19] = 0;
					}
				} else if (SW_SCORE_RESET_BUFFER >= 70) {
					display_score = 1;
					ScoreHeimE = 0;
					ScoreHeimZ = 0;
					ScoreGastE = 0;
					ScoreGastZ = 0;
					send_data[13] = get7[0];
					send_data[15] = get7[0];
					send_data[17] = get7[0];
					send_data[19] = get7[0];
					SW_SCORE_RESET_BUFFER++;
				} else {
					SW_SCORE_RESET_BUFFER++;
				}
			} else {
				SW_SCORE_RESET_BUFFER = 0;
			}
			if (((SW_SIRENE == 0) && (SW_SIRENE_BUFFER >= 5)) || (sirene == 1)) {
				RELAIS = 1;
			} else if ((SW_SIRENE == 0) && (SW_SIRENE_BUFFER < 5)) {
				SW_SIRENE_BUFFER++;
				RELAIS = 0;
			} else if (SW_SIRENE == 1) {
				SW_SIRENE_BUFFER = 0;
				RELAIS = 0;
			}

			// Entprellung
			if (SW_START_STOP_BUFFER > 0) {
				SW_START_STOP_BUFFER--;
			}
			if (SW_HEIM_PLUS_BUFFER > 0) {
				SW_HEIM_PLUS_BUFFER--;
			}
			if (SW_HEIM_MINUS_BUFFER > 0) {
				SW_HEIM_MINUS_BUFFER--;
			}
			if (SW_GAST_PLUS_BUFFER > 0) {
				SW_GAST_PLUS_BUFFER--;
			}
			if (SW_GAST_MINUS_BUFFER > 0) {
				SW_GAST_MINUS_BUFFER--;
			}

			// Steuerung der Dezimalpunkte
			if (running == 1) {
				COLON = 1;
				upper_colon = 1;
				lower_colon = 1;
			}

			// Displaydaten die von der Zeit ahngen laufend aktualisieren
			send_data[1] = get7[SekundenE];
			send_data[3] = get7[SekundenZ];
			send_data[5] = get7[MinutenE];
			send_data[7] = get7[MinutenZ];		
			send_data[9] = upper_colon;
			send_data[11] = lower_colon;

			// Multiplexdaten ermitteln
			display_min = send_data[5 + (mux_status << 1)];
			display_sec = send_data[1 + (mux_status << 1)];
			display_heim = display_score * send_data[13 + (mux_status << 1)];
			display_gast = display_score * send_data[17 + (mux_status << 1)];
			display_drittel = send_data[21 + mux_status];
	
			// Daten an Schieberegister senden
			for (i = 7; i >= 0; i--) {
				DATA = (display_gast ^ (1 << i)) >> i;
				CLOCK = 1;
				CLOCK = 0;
			}
			for (i = 7; i >= 0; i--) {
				DATA = (display_sec ^ (1 << i)) >> i;
				CLOCK = 1;
				CLOCK = 0;
			}
			for (i = 7; i >= 0; i--) {
				DATA = (display_drittel ^ (1 << i)) >> i;
				CLOCK = 1;
				CLOCK = 0;
			}
			for (i = 7; i >= 0; i--) {
				DATA = (display_min ^ (1 << i)) >> i;
				CLOCK = 1;
				CLOCK = 0;
			}
			for (i = 7; i >= 0; i--) {
				DATA = (display_heim ^ (1 << i)) >> i;
				CLOCK = 1;
				CLOCK = 0;
			}

			// Multiplex ausschalten
			MUX_LEFT = 1;
			MUX_RIGHT = 1;

			// Daten durchschalten
			STROBE = 1;
			STROBE = 0;
	
			// Richtigen Multiplexkanal auswhlen
			MUX_LEFT = mux_status;
			MUX_RIGHT = mux_status ^ 1;

			// und JETZT erst den Multiplexkanal wechseln
			mux_status ^= 1;


		// Fernbedienung ist AUS
		} else {

			// nur Nullen an die Schieberegister senden
			for (i = 0; i < 40; i++) {
				DATA = 1;
				CLOCK = 1;
				CLOCK = 0;
			}
			STROBE = 1;
			STROBE = 0;

			// Doppelpunkt und Relais sind auch aus
			COLON = 0;
			RELAIS = 0;

			// Vorbereitender Variablen-Reset, wenn die Fernbedienung eingeschaltet wird

			// Spielzeit pausiert
			running = 0;

			// Spielstand wird angezeigt
			display_score = 1;

			// alle Timebases fangen bei null an
			timebase = 0;
			alarm_timebase = 0;
			blink_timebase = 0;

			// Die Startzeit sollte schon stimmen
			MinutenE = getTime() % 10;
			MinutenZ = getTime() / 10;

			// Die Sekunden sowie die Spielstnde sind satandardmig bei Null
			SekundenE = 0;
			SekundenZ = 0;
			ScoreHeimE = 0;
			ScoreHeimZ = 0;
			ScoreGastE = 0;
			ScoreGastZ = 0;

			// Auch die Gesamtdisplays sollen dann sofort das richtige anzeigen
			// (Wichtig: Momentan senden sie nicht, weil MASTER_EIN == 0 ist)
			send_data[1] = get7[SekundenE];
			send_data[3] = get7[SekundenZ];
			send_data[5] = get7[MinutenE];
			send_data[7] = get7[MinutenZ];
			send_data[9] = upper_colon;
			send_data[11] = lower_colon;
			send_data[13] = get7[ScoreHeimE];
			send_data[15] = get7[ScoreHeimZ];
			send_data[17] = get7[ScoreGastE];
			send_data[19] = get7[ScoreGastZ];
			send_data[21] = get7[Drittel];

		}

	}

}

// ************************************************************************************************
// Interrupt-Service-Routine (ISR)
static void interrupt isr (void) {

	// Timer0-berlauf; bei einem 4194304MHz-Quarz entsprechen 4094 berlufe genau einer Sekunde
	if (T0IF) {

		// Timebase erhhen
		timebase++;

		// Sekunde extrahieren
		if (timebase >= 4095) {

			timebase = 0;

			// Zeit dekrementieren
			if (running == 1) {
				if (SekundenE > 0) {
					SekundenE--;
				} else if ((MinutenE > 0) || (MinutenZ > 0) || (SekundenZ > 0)) {
					SekundenE = 9;
					if (SekundenZ > 0) {
						SekundenZ--;
					} else {
						SekundenZ = 5;
						if (MinutenE > 0) {
							MinutenE--;
						} else {
							if (MinutenZ > 0) {
								MinutenE = 9;
								MinutenZ--;
							}
						}
					}
				}

				// Relais auslsen bei Spielende?
				if (((SekundenE | SekundenZ | MinutenE | MinutenZ) == 0) && (SIRENE_AUTO == 0)) {
					if (alarm_timebase < alarm_time) {
						alarm_timebase++;
						sirene = 1;
					} else {
						sirene = 0;
					}
				} else {
					sirene = 0;
				}

			}

		}
		
		// Das Blinken des Doppelpunktes mit 2Hz realisieren
		blink_timebase++;
		if ((running == 0) && (blink_timebase >= 2045) && (MASTER_EIN)) {
			blink_timebase = 0;
			COLON ^= 1;
			upper_colon ^= 1;
			lower_colon ^= 1;
		}

		// Daten senden, aber nur, wenn die Fernbedienung eingeschaltet ist
		if (MASTER_EIN) {

			// Laufindex erhhen
			what_to_send++;
			if (what_to_send >= 22) {
				what_to_send = 0;
			}

			// Ein Schreiben von einem Wert in das TXREG-Register veranlasst die USART, die Daten zu senden
			// (Bei 128000 Baud geht das so schnell, dass das bei jedem Timer0-Overflow gemacht werden kann)
			TXREG = send_data[what_to_send];

		}

		// Wichtig: Interrupt-Flag resetten
		T0IF = 0;

	}

}

// ************************************************************************************************
// Diese Funktion errechnet die eingestellte Spielzeit aus dem Stand des Spielzeit-Drehschalters
unsigned char getTime () {

	if (TIME_1) {
		return 5;
	} else if (TIME_2) {
		return 10;
	} else if (TIME_3) {
		return 15;
	} else if (TIME_4) {
		return 20;
	} else if (TIME_5) {
		return 25;
	} else if (TIME_6) {
		return 30;
	} else if (TIME_7) {
		return 35;
	} else if (TIME_8) {
		return 40;
	} else if (TIME_9) {
		return 45;
	} else if (TIME_10) {
		return 50;
	} else if (TIME_11) {
		return 55;
	} else {
		return 60;
	}

}