// ************************************************************************************************
//
// Copyright www.jb-electronics.de
//
// Mini-Sport-Display
//
// unter Verwendung des PIC16F716
//
// (x) Anzeige der verbleibenden Spielzeit im Format MM:SS; voreingestellt: 20 Minuten; in Zeile 99f einstellbar
// (x) Start/Stop und Reset der Spielzeit ber je einen Taster
// (x) Signalisieren des Spielzeitendes durch einen Summer; Signallnge ist 1 Sekunde (alternativ in Zeile 105 einstellbar)
// (x) Anzeige des Gast- und Heim-Punktestandes ber je eine einstellige 7-Segment-Anzeige
// (x) Inkrementieren der Gast- und Heim-Punktestnde ber je einen Grobhandtaster ("Buzzer")
// (x) Bei nicht angesteckten Start/Stop-Tastern blinkt die Zeitanzeige
// (x) Bei nicht angesteckten Buzzern blinkt das jeweilige Spielstandsdisplay
//
// http://www.jb-electronics.de/html/elektronik/digital/d_mini_sport_display.htm
//
// ************************************************************************************************


// ************************************************************************************************
// PIC-Header einbinden
#include <pic.h>


// ************************************************************************************************
// Globale Variablen
static int i = 0, ticks = 0, blink_ticks = 0;
static unsigned char Sekunden = 0, Minuten = 0, Score1 = 0, Score2 = 0;
static int SCORE1_BUFFER = 0, SCORE2_BUFFER = 0, START_STOP_BUFFER = 0, RESET_BUFFER = 0, ALARM_BUFFER = 0;
static bit running = 0, colon = 0, alarm = 0, blink = 0, blink_time = 0, blink_score1 = 0, blink_score2 = 0, tmp1 = 0, tmp2 = 0, tmp3 = 0;

								//    fedc.gba
static unsigned char get7[10] = {	0b11110011,
									0b00010010,
									0b01100111,
									0b00110111,
									0b10010110,
									0b10110101,
									0b11110101,
									0b00010011,
									0b11110111,
									0b10110111	};


// ************************************************************************************************
// Hauptmethode
void main (void) {


    // *****
	// Konfigurationsbits setzen
	__CONFIG(FOSC_XT & WDTE_OFF & PWRTE_ON & BOREN_ON & BODENV_40 & CP_OFF);


	// *****
	// TRISTATE-Register setzen, sowie Pindefinitionen
	
	// Anschlsse zu den Schieberegistern
	#define DATA					RA4
	TRISA4 = 0;
	#define CLK						RA2
	TRISA2 = 0;
	#define STROBE_TIME				RA3
	TRISA3 = 0;
	#define STROBE_SCORE1			RB7
	TRISB7 = 0;
	#define STROBE_SCORE2			RB0
	TRISB0 = 0;

	// Tastereingnge zum Einstellen des Heim- und Gast-Punktestands
	#define INC_SCORE1				RB5
	TRISB5 = 1;
	#define INC_SCORE2				RB2
	TRISB2 = 1;

	// Tastereingnge zum Starten/Stoppen/Resetten der Spielzeit
	#define START_STOP				RA1
	TRISA1 = 1;
	#define RESET					RA0
	TRISA0 = 1;

	// Sind die Taster fr das Zeitboard angeschlossen?
	#define START_STOP_CONNECTED	RB4
	TRISB4 = 1;
	#define RESET_CONNECTED			RB3
	TRISB3 = 1;
	
	// Sind die Taster fr den Punktestand angeschlossen?
	#define SCORE1_CONNECTED		RB6
	TRISB6 = 1;
	#define SCORE2_CONNECTED		RB1
	TRISB1 = 1;


	// *****
	// Startzeit setzen
	#define StartMinuten			20
	#define StartSekunden			0


	// *****
	// Lnge der Pausensirene setzen (1 Sekunde entspricht beim 4.194304MHz-Quarz exakt 4096 Ticks; hier also 1,5 Sekunden)
	#define MAX_ALARM_TICKS			6144


	// *****
	// ADC abschalten
	//PCFG0 = 1;
	PCFG1 = 1;
	PCFG2 = 1;
	//ADON = 0;


	// *****
	// TIMER0 einschalten
	T0CS = 0;
	PSA = 1;
	PS0 = 1;
	PS1 = 1;
	PS2 = 1;
	T0IE = 1;


	// *****
	// Interrupts aktivieren

	// global
	GIE = 1;


	// *****
	// Variablen resetten
	running = 0;
	Sekunden = StartSekunden;
	Minuten = StartMinuten;
	Score1 = 0;
	Score2 = 0;
	alarm = 0;
	colon = 1;


	// *****
	// Beginn der Hauptschleife
	while (1) {

		// Blinkstatus (fr gleichzeitiges Blinken) ermitteln
		tmp1 = (blink_time & blink) | (blink_time == 0);
		tmp2 = (blink_score1 & blink) | (blink_score1 == 0);
		tmp3 = (blink_score2 & blink) | (blink_score2 == 0);

		// Zeitdaten senden
		for (i = 7; i >= 0; i--) {
			if (tmp1) {
				DATA = ((get7[Minuten / 10] & (1 << i)) >> i);
			} else {
				DATA = 0;
			}
			CLK = 1;
			CLK = 0;
		}
		for (i = 7; i >= 0; i--) {
			if (tmp1) {
				DATA = ((get7[Minuten % 10] | colon << 3) & (1 << i)) >> i;
			} else {
				DATA = 0;
			}
			CLK = 1;
			CLK = 0;
		}
		for (i = 7; i >= 0; i--) {
			if (tmp1) {
				DATA = ((get7[Sekunden / 10] | alarm << 3) & (1 << i)) >> i;
			} else {
				DATA = 0;
			}
			CLK = 1;
			CLK = 0;
		}
		for (i = 7; i >= 0; i--) {
			if (tmp1) {
				DATA = (get7[Sekunden % 10] & (1 << i)) >> i;
			} else {
				DATA = 0;
			}
			CLK = 1;
			CLK = 0;
		}
		STROBE_TIME = 1;
		STROBE_TIME = 0;


		// Spielstandsdaten senden
		for (i = 7; i >= 0; i--) {
			if (tmp2) {
				DATA = (get7[Score1] & (1 << i)) >> i;
			} else {
				DATA = 0;
			}
			CLK = 1;
			CLK = 0;
		}
		STROBE_SCORE1 = 1;
		STROBE_SCORE1 = 0;
		for (i = 7; i >= 0; i--) {
			if (tmp3) {
				DATA = (get7[Score2] & (1 << i)) >> i;
			} else {
				DATA = 0;
			}
			CLK = 1;
			CLK = 0;
		}
		STROBE_SCORE2 = 1;
		STROBE_SCORE2 = 0;


		// Taster zum Starten/Stoppen bzw. Resetten der Spielzeit angeschlossen?
		if ((START_STOP_CONNECTED == 0) && (RESET_CONNECTED == 0)) {
			blink_time = 0;
		} else {
			blink_time = 1;			
		}

		// Taster zum Erhhen des 1. Punktestandes angeschlossen?
		if (SCORE1_CONNECTED == 1) {
			blink_score1 = 1;
		} else {
			blink_score1 = 0;			
		}

		// Taster zum Erhhen des 2. Punktestandes angeschlossen?
		if (SCORE2_CONNECTED == 1) {
			blink_score2 = 1;
		} else {
			blink_score2 = 0;			
		}

		// 1. Punktestand erhhen?
		if ((INC_SCORE1 == 0) && (SCORE1_BUFFER == 0) && (SCORE1_CONNECTED == 0)) {
			SCORE1_BUFFER = 2000;
			Score1++;
			if (Score1 >= 10) {
				Score1 = 0;
			}
		}

		// 2. Punktestand erhhen?
		if ((INC_SCORE2 == 0) && (SCORE2_BUFFER == 0) && (SCORE2_CONNECTED == 0)) {
			SCORE2_BUFFER = 2000;
			Score2++;
			if (Score2 >= 10) {
				Score2 = 0;
			}
		}

		// Starten bzw. stoppen?
		if ((START_STOP == 0) && (START_STOP_BUFFER == 0) && (START_STOP_CONNECTED == 0)) {
			running ^= 1;
			START_STOP_BUFFER = 2000;
		}

		// Resetten?
		if ((RESET == 0) && (RESET_BUFFER == 0) && (RESET_CONNECTED == 0)) {
			ticks = 0;
			Minuten = StartMinuten;
			Sekunden = StartSekunden;
			running = 0;
			colon = 1;
			alarm = 0;
			ALARM_BUFFER = 0;
			RESET_BUFFER = 2000;
		}


	}


}

// ************************************************************************************************
// Interrupt-Service-Routine (ISR)
static void interrupt isr (void) {

	// TIMER0-berlauf?
	if (T0IF) {

		blink_ticks++;
		if (blink_ticks == 2048) {
			blink ^= 1;
		} else if (blink_ticks >= 4096) {
			blink ^= 1;
			blink_ticks = 0;
		}

		if (running) {
			ticks++;
		}
		if (ticks == 2048) {
			colon ^= 1;
		} else if (ticks >= 4096) {
			colon ^= 1;
			if (Sekunden > 0) {
				Sekunden--;
			} else if (Minuten > 0) {
				Minuten--;
				Sekunden = 59;
			} else {
				Sekunden = 0;
			}
			ticks = 0;
		}
		if ((Minuten == 0) && (Sekunden == 0)) {
			if (ALARM_BUFFER < MAX_ALARM_TICKS) {
				ALARM_BUFFER++;
				running = 0;
				alarm = 1;
			} else {
				alarm = 0;
			}
		}

		// Tasten entprellen
		if (SCORE1_BUFFER > 0) {
			SCORE1_BUFFER--;
		}
		if (SCORE2_BUFFER > 0) {
			SCORE2_BUFFER--;
		}
		if (START_STOP_BUFFER > 0) {
			START_STOP_BUFFER--;
		}
		if (RESET_BUFFER > 0) {
			RESET_BUFFER--;
		}

		T0IF = 0;


	}

}