// ************************************************************************************************
//
// Copyright www.jb-electronics.de
//
// LED-Countdown II
//
// (x) Herunterzhlen eines zweistelligen Ausgangswertes im Sekundentakt
// (x) Einstellbare Startzeit
// (x) Einstellbarer Anfangsoffset zur Synchronisation mit externen Events
// (x) In zehn Stufen einstellbare Helligkeit der LEDs
//
// http://www.jb-electronics.de/html/elektronik/digital/d_countdown2.htm
//
// ************************************************************************************************

// ************************************************************************************************
// PIC-Header einbinden
#include <pic.h>

// ************************************************************************************************
// Methodenprototypen
void sendValues (unsigned char ones, unsigned char tens);
unsigned char calcValue (unsigned d8, unsigned char d4, unsigned char d2, unsigned char d1);

// ************************************************************************************************
// Globale Variablen
static unsigned char count, ones, tens, brightness, offset, index;
static int timecounter = 0, firsttimebuffer = 0;

// Segmentmuster
							//	 __cdegfab_	
static unsigned char get7[10] = {0b11101110,
								 0b10000010,
								 0b01110110,
								 0b11010110,
								 0b10011010,
								 0b11011100,
								 0b11111100,
								 0b10000110,
								 0b11111110,
								 0b11011110};

// Helligkeitstabelle
static unsigned char getPWM[10] = {1, 2, 5, 10, 20, 40, 50, 100, 160, 255};

// ************************************************************************************************
// Hauptmethode
void main (void) {


    // *****
	// Konfigurationsbits setzen
	__CONFIG(FOSC_XT & WDTE_OFF & nPWRTE_ON & MCLRE_ON & BOREN_ON & BORV_25 & CP_OFF);


	// *****
	// Portspezifikationen
	
	#define TENS_1				RC7
	#define TENS_2				RC6
	#define TENS_4				RC5
	#define TENS_8				RC4

	#define ONES_1				RB3
	#define ONES_2				RB2
	#define ONES_4				RB1
	#define ONES_8				RB0

	ANSB0 = 0;
	ANSB1 = 0;
	ANSB2 = 0;
	ANSB3 = 0;

	#define OFFSET_1			RA0
	#define OFFSET_2			RA1
	#define OFFSET_4			RA2
	#define OFFSET_8			RA3

	ANSA0 = 0;
	ANSA1 = 0;
	ANSA2 = 0;
	ANSA3 = 0;

	#define BRIGHTNESS_1		RB7
	#define BRIGHTNESS_2		RB6
	#define BRIGHTNESS_4		RB5
	#define BRIGHTNESS_8		RB4

	ANSB4 = 0;
	ANSB5 = 0;
	
	#define STROBE				RC0
	#define DATA				RC1
	#define PWM					RC2
	#define CLOCK				RC3

	TRISA = 0b1111;
	TRISB = 0b11111111;
	TRISC = 0b11110000;


	// *****
	// Interrupt bei Timer0-Overflow
	GIE = 1;
	T0CS = 0;
	T0IE = 1;
	PSA = 1;


	// *****
	// PWM-Modus fr CCP1-Modul
	CCP1M2 = 1;
	CCP1M3 = 1;
	PR2 = 0xff;
	TMR2ON = 1;
	CCPR1L = 0;


	// *****
	// Offset-Warten
	offset = calcValue(OFFSET_8, OFFSET_4, OFFSET_2, OFFSET_1);
	firsttimebuffer = 409 * offset;


	// *****
	// Einer- und Zehner-Startwerte einmal (!) abfragen
	tens = calcValue(TENS_8, TENS_4, TENS_2, TENS_1);
	ones = calcValue(ONES_8, ONES_4, ONES_2, ONES_1);
	count = tens * 10 + ones;


	// *****
	// Beginn der Hauptschleife
	while (1) {


		// Helligkeit einstellen
		brightness = calcValue(BRIGHTNESS_8, BRIGHTNESS_4, BRIGHTNESS_2, BRIGHTNESS_1);
		CCPR1L = getPWM[brightness];
		

		// Countdown-Wert updaten
		sendValues (count / 10, count % 10);


	}


}

// ************************************************************************************************
// Interrupt-Service-Routine (ISR)
static void interrupt isr (void) {

	if (T0IF) {

		if (firsttimebuffer > 0) {
			firsttimebuffer--;
		} else {
			timecounter++;
		}
		if (timecounter >= 4095) {
			if (count > 0) {
				count--;
			}
			timecounter = 0;
		}
		T0IF = 0;

	}

}

// ************************************************************************************************
// diese Methode berechnet den Wert eines Einstellreglers
unsigned char calcValue (unsigned d8, unsigned char d4, unsigned char d2, unsigned char d1) {

	return (d8 << 3) + (d4 << 2) + (d2 << 1) + d1;

}

// ************************************************************************************************
// Diese Methode sendet zwei Zahlen ans Display
void sendValues (unsigned char ones, unsigned char tens) {

	STROBE = 0;
	for (index = 0; index <= 7; index++) {
		DATA = (get7[ones] & (1 << index)) >> index;
		CLOCK = 1;
		CLOCK = 0;
	}
	for (index = 0; index <= 7; index++) {
		DATA = (get7[tens] & (1 << index)) >> index;
		CLOCK = 1;
		CLOCK = 0;
	}
	STROBE = 1;
	STROBE = 0;

}