// ************************************************************************************************
//
// LED-Countdown unter Verwendung des
//
// PIC 16F627A
//
// Ausgabe der Zeit ber drei LED-Displays, zustzliche Ausgabe ber zwei Doppelpunkte
//
// Starten/Stoppen und Resetten des Countdowns ber zwei Taster
//
// Countdown-Zeit: 5:00 Minuten
//
// Copyright www.jb-electronics.de
//
// ************************************************************************************************

// ************************************************************************************************
// PIC-Header einbinden
#include <pic.h>

// ************************************************************************************************
// Methodenprototypen
void warte ();
uchar get7SegmentValue (uchar Wert, uchar DecimalPoint);
void schreibeWert (uchar Wert, uchar Wohin);
void updateDisplay ();

// ************************************************************************************************
// Globale Variablen
static int TimingCounter, Sekunden, Active, tmp;

// ************************************************************************************************
// Hauptmethode
void main (void) {

    // ********************************************************************************************
	// Konfigurationsbits setzen
	__CONFIG(0x3F2A);

	// ********************************************************************************************
	// Portspezifikationen
	#define	DATA	RB0
	#define CLK		RB1
	#define SEG1	RB2
	#define SEG2	RB3
	#define SEG3	RB4
	#define COLON	RB5
	#define TOGGLE	RB6
	#define RESET	RB7

	// Tristate setzen
	TRISB0 = 0;
	TRISB1 = 0;
	TRISB2 = 0;
	TRISB3 = 0;
	TRISB4 = 0;
	TRISB5 = 0;
	TRISB6 = 1;
	TRISB7 = 1;

	// ********************************************************************************************
	// TIMER setzen
	T0CS = 0;
	PSA = 1;
	PS0 = 1;
	PS1 = 1;
	PS2 = 1;

	// Interrupt bei Timer-berlauf auslsen
	T0IE = 1;

	// Interrupts aktivieren
	GIE = 1;

	// ********************************************************************************************
	// Countdown-Zeit auf 5 Minuten setzen
	Active = 0;
	Sekunden = 300;

	// ********************************************************************************************
	// Beginn der Hauptschleife
	while (true) {

		// Countdownstand anzeigen
		updateDisplay ();

		// Starten/Stoppen?
		if (TOGGLE) {
			Active^ = 1;
			while (TOGGLE) {
				updateDisplay ();
			}
		}

		// Resetten?
		if (RESET) {
			Active = 0;
			Sekunden = 300;
			while (RESET) {
				updateDisplay ();
			}
		}

	}

}

// ************************************************************************************************
// Interrupt-Routine fr genauen 1Hz-Takt
static void interrupt isr (void) {

	// ********************************************************************************************
	// ein bisschen Zeit verstrichen?
	if (T0IF) {

		TimingCounter++;

		// eine halbe Sekunde um?
		if (TimingCounter == 2048) {
			COLON^ = 1;
		}

		// eine Sekunde um?
		if (TimingCounter == 4095) {
			if (Active) {
				COLON^ = 1;
				Sekunden--;
				if (Sekunden < 0) {
					Sekunden = 0;
				}
			} else {
				COLON = 1;
			}
			TimingCounter = 0;
		}

		// Countdown vorbei?
		if (Sekunden == 0) {
			COLON = 1;
		}

		// Interrupt resetten
		T0IF = 0;

	}

}

// ************************************************************************************************
// Diese Methode wartet kurz, um Signale gltig werden zu lassen
void warte () {

	// einen Takt lang nichts tun
	NOP();

}

// ************************************************************************************************
// schreibt einen Wert in ein Schieberegister
void schreibeWert (uchar Wert, uchar Wohin) {

	// Variablen
	int Index;

	// initialisieren
	SEG1 = 0;
	SEG2 = 0;
	SEG3 = 0;

	// erst die Zehner
	for (Index = 7; Index >= 0; Index--) {
		warte();
		if (Wert & (1 << Index)) {
			DATA = 1;
		} else {
		 	DATA = 0;
		}
		warte();
		CLK = 1;
		warte();
		CLK = 0;
	}

	// Wert bernehmen
	if (Wohin == 1) {
		SEG1 = 1;
	} else if (Wohin == 2) {
		SEG2 = 1;
	} else if (Wohin == 3) {
		SEG3 = 1;
	}
	warte();

	// zurcksetzen
	SEG1 = 0;
	SEG2 = 0;
	SEG3 = 0;

}

// ************************************************************************************************
// wandelt einen Wert in ein Zeichen im 7-Segment-Code um
uchar get7SegmentValue (uchar Wert, uchar DecimalPoint) {

	// Variablen
	uchar result;

	// welches Ausgabemuster bei welchem Zeichen?
	//
	//   Bit    Segment
	//
	//    0       a
	//    1       b
	//    2       c
	//    3       d
	//    4       f
	//    5       g
	//    6       h
	//    7       DP
	//
	if (Wert == 0) {
		result = 0b00111111;
	} else if (Wert == 1) {
		result = 0b00000110;
	} else if (Wert == 2) {
		result = 0b01011011;
	} else if (Wert == 3) {
		result = 0b01001111;
	} else if (Wert == 4) {
		result = 0b01100110;
	} else if (Wert == 5) {
		result = 0b01101101;
	} else if (Wert == 6) {
		result = 0b01111101;
	} else if (Wert == 7) {
		result = 0b00000111;
	} else if (Wert == 8) {
		result = 0b01111111;
	} else if (Wert == 9) {
		result = 0b01101111;
	} else {
		result = 0b01110001;
	}

	// Dezimalpunkt aktiv?
	if (DecimalPoint) {
		result| = 0b10000000;
	}
	
	// Wert zurckgeben
	return (result);

}

// ************************************************************************************************
// Diese Methode zeigt den Countdownstand an
void updateDisplay () {

	// Countdownstand anzeigen
	schreibeWert(get7SegmentValue(Sekunden / 60, 0), 1);
	schreibeWert(get7SegmentValue((Sekunden - (Sekunden / 60) * 60) / 10, 0), 2);
	schreibeWert(get7SegmentValue((Sekunden - (Sekunden / 60) * 60) % 10, 0), 3);

}
