// ************************************************************************************************
//
// LCD-Ansteuerung unter Verwendung des
//
// PIC 16F627A
//
// Copyright www.jb-electronics.de
//
// ************************************************************************************************

// ************************************************************************************************
// PIC-Header einbinden
#include <pic.h>

// ************************************************************************************************
// Methodenprototypen
void warte ();
void sendeWert (uchar Wert);
void initLCD ();
void updateLCD ();
void setCursor (uchar DISPLAY_ON, uchar CURSOR_ON, uchar BLINKING);
void setCursorHome ();
void setEntryMode (uchar MOVE_RIGHT, uchar SHIFT);
void setFunction (uchar INTERFACE_8BIT, uchar TWOROWS, uchar MATRIX_5_7);
void setShift (uchar MOVE_DISPLAY, uchar MOVE_RIGHT);
void clearDisplay ();
void sendCommand (int Command);
void writeChar (uchar Zeichen, uchar Row, uchar Column);
void writeCharBuffered (uchar Zeichen, uchar Row, uchar Column);
void writeTextBuffered (uchar Text[], uchar Row, uchar Column, uchar Length);
void setAddressDDRAM (uchar Address);
void setPosition (uchar Row, uchar Column);

// ************************************************************************************************
// Die Textdaten
static bank1 uchar Zeile1[16] = {' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '};
static bank1 uchar Zeile2[16] = {' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '};

// ************************************************************************************************
// Die Textdaten
static uchar BeispielText[13] = {'B', 'e', 'i', 's', 'p', 'i', 'e', 'l', '-', 'T', 'e', 'x', 't'};
static uchar SekundenText[8] = {'S', 'e', 'k', 'u', 'n', 'd', 'e', 'n'};

// ************************************************************************************************
// Sonstige Variablen

static int TimeCounter, Sekunden, w;

// ************************************************************************************************
// Hauptmethode
void main (void) {

    // ********************************************************************************************
	// Konfigurationsbits setzen
	__CONFIG(0x3F2A);

	// ********************************************************************************************
	// Portspezifikationen

	#define CLK		RB0
	#define	DATA	RB1
	#define STROBE	RB2
	#define LICHT	RB3
	#define RS		RA0
	#define RW		RA1
	#define ENABLE	RA2

	TRISB = 0b00000000;
	TRISA0 = 0b0;
	TRISA1 = 0b0;
	TRISA2 = 0b0;

	// Analogkomparator bei PORTA abschalten
	CMCON = 0b111;

	// ********************************************************************************************
	// TIMER setzen
	T0CS = 0;
	PSA = 1;
	PS0 = 1;
	PS1 = 1;
	PS2 = 1;

	// Interrupt bei Timer-berlauf auslsen
	T0IE = 1;

	// Interrupts aktivieren
	GIE = 1;

	// LCD initialisieren und warten
	initLCD();
	for (w = 0; w < 200; w++) {
		NOP();
	}

	// ********************************************************************************************
	// LCD-Beleuchtung EIN
	LICHT = 1;

	// ********************************************************************************************
	// Beginn der Hauptschleife
	while (true) {


		// ****************************************************************************************
		// Anzeigen der entsprechenden Inhalte

		// Texte anzeigen
		writeTextBuffered (BeispielText, 1, 1, 13);
		writeTextBuffered (SekundenText, 2, 7, 8);

		// Sekunden anzeigen
		writeCharBuffered (48 + Sekunden / 100, 2, 3);
		writeCharBuffered (48 + Sekunden / 10 - (Sekunden / 100) * 10, 2, 4);
		writeCharBuffered (48 + Sekunden % 10, 2, 5);

		


		// ****************************************************************************************
		// Die gepufferten Daten zu LCD senden
		updateLCD ();


	}

}

// ************************************************************************************************
// Interrupt-Routine fr genauen 1Hz-Takt
static void interrupt isr (void) {

	// ********************************************************************************************
	// ein bisschen Zeit verstrichen?
	if (T0IF) {

		TimeCounter++;

		if (TimeCounter == 3906) {
			TimeCounter = 0;
			Sekunden++;
		}

		T0IF = 0;

	}

}

// ************************************************************************************************
// Diese Methode wartet kurz, um Signale gltig werden zu lassen
void warte () {

	for (w = 0; w < 5; w++) {
		NOP();
	}

}

// ************************************************************************************************
// Diese Methode initialisiert das LCD
void initLCD () {

	// warten
	int i;
	ENABLE = 0;
	for (i = 0; i < 5000; i++) {
		NOP();
	}

	// Cursor unsichtbar
	setCursor(1, 0, 0);

	// 2zeilig
	setFunction (1, 1, 1);

	// Cursor auf Ausgangsposition
	setCursorHome ();

	// Display lschen
	clearDisplay();

}

// ************************************************************************************************
// Diese Methode setzt die aktuelle Adresse im DDRAM
void setAddressDDRAM (uchar Address) {

	sendCommand (128 + Address);

}

// ************************************************************************************************
// Diese Methode setzt die Cursoreinstellungen
void setCursor (uchar DISPLAY_ON, uchar CURSOR_ON, uchar BLINKING) {

	sendCommand(8 + (DISPLAY_ON << 2) + (CURSOR_ON << 1) + BLINKING);

}

// ************************************************************************************************
// Diese Methode setzt die Displayeinstellungen
void setFunction (uchar INTERFACE_8BIT, uchar TWOROWS, uchar MATRIX_5_7) {

	sendCommand(32 + (INTERFACE_8BIT << 4) + (TWOROWS << 3) + (MATRIX_5_7 << 2) + 2 + 1);

}

// ************************************************************************************************
// Diese Methode lscht das Display
void clearDisplay () {

	sendCommand(1);
	for (w = 0; w < 200; w++) {
		NOP();
	}

}

// ************************************************************************************************
// Diese Methode setzt den Cursor auf die Ursprungsposition
void setCursorHome () {

	sendCommand(3);

}

// ************************************************************************************************
// Diese Methode setzt Eingabemodus des Displays
void setEntryMode (uchar MOVE_RIGHT, uchar SHIFT) {

	sendCommand(4 + (MOVE_RIGHT << 1) + SHIFT);

}

// ************************************************************************************************
// Diese Methode setzt den Shift des Displays
void setShift (uchar MOVE_DISPLAY, uchar MOVE_RIGHT) {

	sendCommand(16 + (MOVE_DISPLAY << 3) + (MOVE_RIGHT << 2) + 2 + 1);

}

// ************************************************************************************************
// Diese Methode sendet ein Kommando an das LCD
void sendCommand (int Command) {

	RW = 0;
	RS = 0;
	sendeWert (Command);
	ENABLE = 1;
	NOP();
	ENABLE = 0;
	warte ();

}

// ************************************************************************************************
// Diese Methode sendet ein Zeichen in den PIC-Zeichepuffer
void writeCharBuffered (uchar Zeichen, uchar Row, uchar Column) {

	if (Row == 1) {
		Zeile1[Column - 1] = Zeichen;
	} else if (Row == 2) {
		Zeile2[Column - 1] = Zeichen;
	}

}

// ************************************************************************************************
// Diese Methode sendet ein Zeichen direkt an das LCD
void writeChar (uchar Zeichen, uchar Row, uchar Column) {

	// Cursor-Position setzen
	setPosition (Row, Column);

	// Zeichen senden
	RW = 0;
	RS = 1;
	sendeWert (Zeichen);
	ENABLE = 1;
	NOP();
	ENABLE = 0;
	warte ();

}

// ************************************************************************************************
// Diese Methode schreibt eien Text an das Display
void writeTextBuffered (uchar Text[], uchar Row, uchar Column, uchar Length) {

	uchar t;
	for (t = 0; t < Length; t++) {
		writeCharBuffered (Text[t], Row, Column + t);
	}

}

// ************************************************************************************************
// Diese Methode schreibt alle Daten aus dem PIC-Zeichenpuffer in das LCD
void updateLCD () {

	uchar t;
	for (t = 1; t <= 16; t++) {
		writeChar (Zeile1[t - 1], 1, t);
		writeChar (Zeile2[t - 1], 2, t);
	}

}

// ************************************************************************************************
// Diese Methode platziert den Cursor
void setPosition (uchar Row, uchar Column) {

	if (Row == 1) {
		setAddressDDRAM(Column - 1);
	} else {
		setAddressDDRAM(64 + Column - 1);
	}

}

// ************************************************************************************************
// Diese Methode sendet einen Wert (binr zerlegt) an ein Schieberegister
void sendeWert (uchar Wert) {

	// Variablen
	int Index;

	// Wert senden
	for (Index = 7; Index >= 0; Index--) {
		if (Wert & (1 << Index)) {
			DATA = 1;
		} else {
		 	DATA = 0;
		}
		CLK = 1;
		CLK = 0;
	}

	STROBE = 1;
	STROBE = 0;

}
