// ************************************************************************************************
//
// Electronic Candle
//
// PIC16F716
//
// Copyright www.jb-electronics.de
//
// http://www.jb-electronics.de/html/elektronik/digital/d_elektronische_kerze.htm
//
// ************************************************************************************************

// ************************************************************************************************
// include pic header
#include <pic.h>

// ************************************************************************************************
// method prototypes
unsigned char getRandomNumber ();

// ************************************************************************************************
// global variables
static int t = 0, newValue = 0, maxValue = 0, mean_values = 0, speed = 0, fade_delay = 0, rnd = 0;
static unsigned char random_number = 0, mode = 0;

// ************************************************************************************************
// main method
void main (void) {


    // *****
	// configuration bit
	__CONFIG(0x3f29);


	// *****
	// port specifications
    #define CANDLE	RB3
	TRISB3 = 0;

    #define ADC_0	RA0
	#define ADC_1	RA1
	#define ADC_2	RA2
	#define ADC_3	RA3
	TRISA = 0b1111;

	#define MODUS_A	RB1
	#define MODUS_B	RB4
	#define MODUS_C	RB0
	#define MODUS_D	RB2
	
	// enable weak internal pullups on PORTB
	nRBPU = 0;


	// *****
	// ADC settings:

	// ADC sampling time per bit is 32 * T_osc
	ADCS0 = 0;
	ADCS1 = 1;

	// RA0:3 ports as analog inputs
	PCFG0 = 0;
	PCFG1 = 0;
	PCFG2 = 0;

	// turn it ON
	ADON = 1;

	// wait some time
	for (t = 0; t <= 100; t++) {
		NOP ();
	}


	// *****
	// configure Timer0 as a timer with no prescaler, interrupt on overflow
	T0CS = 0;
	PSA = 1;
	T0IE = 1;

	// enable global interrupts
	GIE = 1;


	// *****
	// PWM settings

	// enable PWM mode
	CCP1M0 = 0;
	CCP1M1 = 0;
	CCP1M2 = 1;
	CCP1M3 = 1;

	// only RB3 as an output, all other pins as digital IOs
	P1M0 = 0;
	P1M1 = 0;

	
	// PWM duty cycle

	// (two LSB of 10 Bit value)
	DC1B0 = 0;
	DC1B1 = 0;

	// (eight MSB if 10 Bit value)
	CCPR1L = 0;

	// upper limit of Timer2
	PR2 = 0xff;

	// set prescaler of Timer2 to 1:1
	T2CKPS0 = 0;
	T2CKPS1 = 0;

	// turn it on
	TMR2ON = 1;


	// *****
	// default candle mode
	mode = 1;


	// *****
	// main loop
	while (1) {


		// which candle mode?
		if (mode == 0) {	
			maxValue = 200;
			mean_values = 1;
			fade_delay = 0;
		} else if (mode == 1) {
			maxValue = 200;
			mean_values = 2;
			fade_delay = 4;
		} else if (mode == 2) {	
			maxValue = 200;
			mean_values = 8;
			fade_delay = 4;
		} else if (mode == 3) {
			maxValue = 250;
			mean_values = 16;
			fade_delay = 4;
		} else if (mode == 4) {
			maxValue = 250;
			mean_values = 32;
			fade_delay = 4;
		} else if (mode == 5) {
			maxValue = 400;
			mean_values = 32;
			fade_delay = 4;
		} else if (mode == 6) {
			maxValue = 600;
			mean_values = 32;
			fade_delay = 4;
		} else if (mode == 7) {
			maxValue = 1000;
			mean_values = 32;
			fade_delay = 16;
		} else if (mode == 8) {
			maxValue = 1000;
			mean_values = 64;
			fade_delay = 16;
		} else if (mode == 9) {
			maxValue = 1000;
			mean_values = 128;
			fade_delay = 16;
		}

		// the actual animation
		if (newValue >= maxValue) {

			rnd = 0;
			for (t = 0; t < mean_values; t++) {
				rnd += getRandomNumber();
			}
			random_number = rnd / t;
			newValue = 0;

		}


		// calculate new mode
		mode = (!MODUS_A) | ((!MODUS_B) << 1) | ((!MODUS_C) << 2) | ((!MODUS_D) << 3);


	}


}

// ************************************************************************************************
// interrupt service routine
static void interrupt isr (void) {

	if (T0IF) {

		newValue++;
		speed++;
		if (speed >= fade_delay) {
			if (CCPR1L > random_number) {
				CCPR1L--;
			} else if (CCPR1L < random_number) {
				CCPR1L++;
			}
			speed = 0;
		}

		// reset interrupt flag
		T0IF = 0;

	}

}

// ************************************************************************************************
void waitSomeTime () {

	NOP ();	NOP ();	NOP ();	NOP ();	NOP ();
	NOP ();	NOP ();	NOP ();	NOP ();	NOP ();
	NOP ();	NOP ();	NOP ();	NOP ();	NOP ();
	NOP ();	NOP ();	NOP ();	NOP ();	NOP ();

}

// ************************************************************************************************
void waitSomeLongTime () {

	for (t = 0; t < 5000; t++) {
		waitSomeTime ();
	}

}

// ************************************************************************************************
// this method creates a 4 bit random number by noise on the four ADCs' LSB bits
// and then shifts this number left by four bits
unsigned char getRandomNumber () {

	unsigned char tmp = 0;

	CHS0 = 0; CHS1 = 0; CHS2 = 0; GO = 1; while (GO);
	tmp = ADRES & 1;
	
	CHS0 = 1; CHS1 = 0; CHS2 = 0; GO = 1; while (GO);
	tmp += (ADRES & 1) << 1;
	
	CHS0 = 0; CHS1 = 1; CHS2 = 0; GO = 1; while (GO);
	tmp += (ADRES & 1) << 2;
	
	CHS0 = 1; CHS1 = 1; CHS2 = 0; GO = 1; while (GO);
	tmp += (ADRES & 1) << 3;

	return (tmp << 4);

}